import maya.api.OpenMaya as om
import maya.cmds as cmds
import maya.mel as mel

from PySide6.QtCore import QTimer
from PySide6.QtGui import QPixmap
from PySide6.QtWidgets import QApplication, QWidget, QPushButton, QLineEdit, QVBoxLayout, QLabel

import logging
import os

# Configure logging
logging.basicConfig(
    level=logging.ERROR,)


def maya_useNewAPI():
    pass

VERSION = "1.0.6"
LICENSE = "VSDL"

class StoreAssetCmd(om.MPxCommand):

    COMMAND_NAME = "storeAsset"

    FLAG = ["-ui", "-userInterface"]

    RESULT = None

    def __init__(self):

        super(StoreAssetCmd, self).__init__()

    def doIt(self, args):

        try:
            arg_db = om.MArgDatabase(self.syntax(), args)
        except:
            self.displayError("Flag value must be boolean")
            return

        openUI = arg_db.isFlagSet(StoreAssetCmd.FLAG[0])

        if openUI:
            self.run()
        else:
            self.run_noUI()


        


    @classmethod
    def creator(cls):

        return StoreAssetCmd()

    @classmethod
    def create_syntax(cls):

        syntax = om.MSyntax()

        syntax.addFlag(cls.FLAG[0], cls.FLAG[1])

        return syntax



    def run(self, *args):

        mel.eval('thumbnailCaptureComponent -capture;')

        def save_button():
            active_window = QApplication.activeWindow()
            checkQLabel(active_window)
            imageWidget = active_window.findChild(QLabel, "image")
            if imageWidget == None:
                cmds.error('Error')
                return

            asset_name = active_window.findChild(QLineEdit).text()
            if asset_name == '':
                cmds.warning('No asset name given')
                return
            ws = cmds.workspace(q = True, fullName = True)
            wsp = ws + "/" + "assets/.mayaSwatches"
            cmds.sysFile(wsp, makeDir=True)

            imageSnapshot = wsp + "/" + asset_name + ".mb.swatch"
            imageWidget.pixmap().save(imageSnapshot, "PNG")
            assetFile = cmds.file(ws + "/" + "assets/" + asset_name + ".mb" , es = True,type="mayaBinary")
            #StoreAssetCmd.RESULT = 'Asset saved to ' + str(assetFile)
            om.MGlobal.displayInfo( 'Result: Asset saved to ' + str(assetFile))
            cancel_button()
            

        def cancel_button():
            mel.eval('fileOptionsCancel;')
            active_window = QApplication.activeWindow()
            if active_window:
                active_window.close()

        def list_all_widgets(widget, indent=0, button_count=0, buttons=[]):

            logging.debug(" " * indent + f"{widget.__class__.__name__} - {widget.objectName()}")


            if isinstance(widget, QPushButton):
                button_count += 1
                if button_count == 3:
                    widget.clicked.disconnect()  
                    widget.clicked.connect(save_button) 
                    buttons.append(widget)
                    logging.debug("Button 1 connected")
                elif button_count == 4:
                    widget.clicked.disconnect()  
                    widget.clicked.connect(cancel_button)  
                    buttons.append(widget)
                    logging.debug("Button 2 connected")

            for child in widget.findChildren(QWidget):
                button_count, buttons = list_all_widgets(child, indent + 2, button_count, buttons)

            return button_count, buttons

        def checkQLabel(widget,):
            if isinstance(widget, QLabel):
                if widget.pixmap().size().width() > 180:
                    widget.setObjectName("image")
            for child in widget.findChildren(QWidget):
                checkQLabel(child)


        def add_asset_name_box():

            active_window = QApplication.activeWindow()

            if active_window:

                layout = active_window.layout()
                if layout is None:
                    layout = QVBoxLayout(active_window) 
                    active_window.setLayout(layout)

                existing_widget = active_window.findChild(QLineEdit, "asset_name_box")
                if existing_widget is None:
                    asset_name_box = QLineEdit(active_window)
                    asset_name_box.setObjectName("asset_name_box")
                    asset_name_box.setPlaceholderText("Enter asset name here")
                    asset_name_box.resize(200, 30)  

                    layout.insertWidget(0, asset_name_box)  # This ensures it's placed at the top

                    active_window.update()  # Make sure the window is updated to reflect the new widget

            else:
                logging.debug("No active window")

        def check_active_window():

            active_window = QApplication.activeWindow()
            mel.eval('fileOptionsCancel;')

            if active_window:

                list_all_widgets(active_window)
                
                add_asset_name_box() 
                
            else:
                logging.debug("No active window")


        QTimer.singleShot(6, check_active_window)



    def run_noUI(self, *args):

        def save_camera_attributes(camera_name):

            if not cmds.objExists(camera_name):
                print(f"Camera '{camera_name}' does not exist.")
                return
            

            camera_translate = cmds.getAttr(f"{camera_name}.translate")[0]
            camera_rotate = cmds.getAttr(f"{camera_name}.rotate")[0]

            focal_length = cmds.getAttr(f"{camera_name}.focalLength")
            
            camera_attributes = {
                'translate': camera_translate,
                'rotate': camera_rotate,
                'focalLength': focal_length
            }
            
            return camera_attributes

        camera_name = 'persp'
        camera_data = save_camera_attributes(camera_name)
        selectedAssetList = cmds.ls(sl = True)
        if selectedAssetList == []:
            om.MGlobal.displayError('Nothing is currently selected to export')
            return
        else:
            selectedAsset = selectedAssetList[0]
        
        ws = cmds.workspace(q = True, fullName = True)
        wsp = ws + "/" + "assets/.mayaSwatches"
        cmds.sysFile(wsp, makeDir=True)

        imageSnapshot = wsp + "/" + selectedAsset + ".mb.swatch"

        if os.path.isfile(imageSnapshot):
            cmds.warning('Asset already exists')
            return

        cmds.viewFit(selectedAsset)

        background_color = cmds.displayRGBColor('background', query=True)

        cmds.displayRGBColor('background', 0, 0, 0)

        cmds.select(cl = True)
        imageFile = cmds.playblast(
            startTime=1,
            endTime=1,  
            format="image",  
            filename=imageSnapshot,
            widthHeight=[240, 240],  
            viewer=False, 
            forceOverwrite=True,
            percent=100, 
            quality=100,
            offScreen=True,
            frame=[1]
        )
        imageFile = imageFile.replace('####', '0000')
        swatch_index = imageFile.lower().rfind('swatch')
        new_name = imageFile[:swatch_index + len('swatch')]
        os.rename(imageFile, new_name)

        cmds.select(selectedAsset)
        assetFile = cmds.file(ws + "/" + "assets/" + selectedAsset + ".mb" , es = True,type="mayaBinary")

        cmds.setAttr(f"{camera_name}.translateX", camera_data['translate'][0])
        cmds.setAttr(f"{camera_name}.translateY", camera_data['translate'][1])
        cmds.setAttr(f"{camera_name}.translateZ", camera_data['translate'][2])

        cmds.setAttr(f"{camera_name}.rotateX", camera_data['rotate'][0])
        cmds.setAttr(f"{camera_name}.rotateY", camera_data['rotate'][1])
        cmds.setAttr(f"{camera_name}.rotateZ", camera_data['rotate'][2])

        cmds.setAttr(f"{camera_name}Shape.focalLength", camera_data['focalLength'])

        cmds.displayRGBColor('background', background_color[0], background_color[1], background_color[2])

        #StoreAssetCmd.RESULT = 'Asset saved to ' + str(assetFile)
        om.MGlobal.displayInfo( 'Result: Asset saved to ' + str(assetFile))


def buildMenu():
    mel.eval('buildFileMenu();')
    menus = cmds.window("MayaWindow", query=True, menuArray=True)

    file_menu_items = cmds.menu('mainFileMenu', query=True, itemArray=True)
    storeAsset_menu = cmds.menuItem('storeAsset_menu', label="Store Asset in Project...", parent="mainFileMenu", insertAfter='exportAllFileOptions', c = 'import maya.cmds as cmds;cmds.storeAsset(ui = False)')
    cmds.menuItem(optionBox = True, c = 'import maya.cmds as cmds;cmds.storeAsset(ui = True)', parent="mainFileMenu", insertAfter='storeAsset_menu')

def initializePlugin(plugin):

    vendor = "Nithish Venkidu Samy"
    version = VERSION

    plugin_fn = om.MFnPlugin(plugin, vendor, version)

    try:
        plugin_fn.registerCommand(StoreAssetCmd.COMMAND_NAME, StoreAssetCmd.creator, StoreAssetCmd.create_syntax)
        cmds.evalDeferred(buildMenu)


    except:
        om.MGlobal.displayError("Failed to register command: {0}".format(StoreAssetCmd))

def uninitializePlugin(plugin):

    mel.eval('fileOptionsCancel;')
    plugin_fn = om.MFnPlugin(plugin)
    try:
        menu_item_name = "storeAsset_menu"

        if cmds.menuItem(menu_item_name, exists=True):
            cmds.deleteUI(menu_item_name)
            
        plugin_fn.deregisterCommand(StoreAssetCmd.COMMAND_NAME)
    except:
        om.MGlobal.displayError("Failed to deregister command: {0}".format(StoreAssetCmd))



























if __name__ == "__main__":

    plugin_name = "storeAsset.py"

    cmds.evalDeferred('if cmds.pluginInfo("{0}", q=True, loaded=True): cmds.unloadPlugin("{0}")'.format(plugin_name))
    cmds.evalDeferred('if not cmds.pluginInfo("{0}", q=True, loaded=True): cmds.loadPlugin("{0}")'.format(plugin_name))
